#!/bin/bash

CFG_FILE=/opt/fqlink/can_app/config/fryer.conf
CONNECTION_NAME=FryerWiFiConnection
PARAM_SSID="SSID"
PARAM_PASS="PSK"
PARAM_SECURITY="KEY_MGMT"
WIFI_SECURITY_WPA="wpa-psk"
PARAM_NETWORK_MODE="NETWORK_MODE"
WIFI_MODE="WI-FI"
ETHERNET_MODE="ETHERNET"
DEVICE_NAME="wlan0"

# read the fryer configuration file
unset FRYER_CFG
declare -A FRYER_CFG
while IFS=$'[ \t]*=[ \t]*' read -r name value
do
    if [ ! -z ${name} ]; then
        FRYER_CFG[${name}]="${value}"
    fi
done < ${CFG_FILE}

# Check for the required NETWORK_MODE
if [ -z "${FRYER_CFG[${PARAM_NETWORK_MODE}]}" ]; then
    echo "Error: NETWORK_MODE is not specified" >&2
    exit 1
fi

if [ ${FRYER_CFG[${PARAM_NETWORK_MODE}]} != ${WIFI_MODE} ]; then
    echo "NETWORK_MODE is not configured as WI-FI (${FRYER_CFG[${PARAM_NETWORK_MODE}]})"
    echo "Disable wireless connection"
    nmcli radio wifi off
    exit 0
else
    echo "NETWORK_MODE is configured as WI-FI"
fi

# Check the required parameters
if [ -z "${FRYER_CFG[${PARAM_SSID}]}" ]; then
    echo "Error: SSID is not specified" >&2
    exit 1
fi

if [ -z "${FRYER_CFG[${PARAM_SECURITY}]}" ]; then
    echo "Warning: KEY_MGMT is not specified. Will use open Wi-Fi connection" >&2
else
    if [ ${FRYER_CFG[${PARAM_SECURITY}]} == ${WIFI_SECURITY_WPA} ]; then
        if [ -z "${FRYER_CFG[${PARAM_PASS}]}" ]; then
            echo "Error: PSK (or passphrase) is not specified" >&2
            exit 1
        fi
    fi
fi

# Function to check connection status
check_connection() {
    active_connection=$(nmcli dev | grep ${DEVICE_NAME} | grep -w 'connected' | awk '{$1=$2=$3="";print $0}' | sed -r "s/^\s+//g" 2>/dev/null)
    if [ -n "$active_connection" ]; then
        echo "Current active connection is '${active_connection}'"
        return 0  # Connection is successful
    else
        echo "No active connection"
        return 1  # Connection is not successful
    fi
}

# Deactivate any active Wi-Fi connection before starting a new one
active_connection=$(nmcli dev | grep ${DEVICE_NAME} | grep -w 'connected' | awk '{$1=$2=$3="";print $0}' | sed -r "s/^\s+//g" 2>/dev/null)
if [ -n "$active_connection" ]; then
    echo "Deactivate current connection"
    nmcli con down ${active_connection}
else
    echo "No active Wi-Fi connection found"
fi

# Start the infinite loop to attempt connection until successful
while true; do
    # Check if the connection already exists
    result=$(nmcli con show ${CONNECTION_NAME} 2>/dev/null)
    RES=$?

    if [ $RES -eq 0 ]; then
        echo "Connection '${CONNECTION_NAME}' already exists"
        DEVICE=$(nmcli con | grep ${CONNECTION_NAME} | awk '{print $4}')
        if [ ${DEVICE} == ${DEVICE_NAME} ]; then
            echo "Deactivate connection"
            nmcli con down ${CONNECTION_NAME}
        else
            echo "Connection is already deactivated"
        fi
        echo "Modify existing connection"
        nmcli con modify ${CONNECTION_NAME} ssid "${FRYER_CFG[${PARAM_SSID}]}"
    else
        echo "Create new connection"
        nmcli con add con-name ${CONNECTION_NAME} ifname wlan0 type wifi ssid "${FRYER_CFG[${PARAM_SSID}]}"
    fi

    # Setup security properties
    if [ -z ${FRYER_CFG[${PARAM_SECURITY}]} ] || [ ${FRYER_CFG[${PARAM_SECURITY}]} != ${WIFI_SECURITY_WPA} ]; then
        result=$(nmcli -f 802-11-wireless-security con show ${CONNECTION_NAME} 2>/dev/null)
        if [ -n "$result" ]; then
            echo "Delete security properties"
            nmcli con del ${CONNECTION_NAME}
            nmcli con add con-name ${CONNECTION_NAME} ifname wlan0 type wifi ssid "${FRYER_CFG[${PARAM_SSID}]}"
        fi
    else
        echo "Setup security properties"
        nmcli con modify ${CONNECTION_NAME} wifi-sec.key-mgmt "${FRYER_CFG[${PARAM_SECURITY}]}"
        nmcli con modify ${CONNECTION_NAME} wifi-sec.psk "${FRYER_CFG[${PARAM_PASS}]}"
    fi

    # Enable Wi-Fi radio
    nmcli radio wifi on

    # Activate the Wi-Fi connection
    ip route del default
    nmcli con up id ${CONNECTION_NAME}

    sleep 10

    # Check if the connection was successful
    if check_connection; then
        echo "Wi-Fi connection successful!"
        break  # Exit the loop if connection is successful
    else
        echo "Wi-Fi connection failed, retrying..."
        sleep 30  # Wait for 30 seconds before retrying
    fi

    # Recheck connection before retrying
    if check_connection; then
        echo "Wi-Fi connection successful!"
        break  # Exit the loop if connection is successful
    fi
done

exit 0
